<?php

declare (strict_types=1);
namespace PHPStan\Rules\PhpDoc;

use PhpParser\Node;
use PHPStan\PhpDoc\Tag\RequireExtendsTag;
use PHPStan\Rules\ClassNameCheck;
use PHPStan\Rules\ClassNameNodePair;
use PHPStan\Rules\RuleError;
use PHPStan\Rules\RuleErrorBuilder;
use PHPStan\Type\ObjectType;
use PHPStan\Type\VerbosityLevel;
use function array_merge;
use function count;
use function sprintf;
final class RequireExtendsCheck
{
    /**
     * @var ClassNameCheck
     */
    private $classCheck;
    /**
     * @var bool
     */
    private $checkClassCaseSensitivity;
    public function __construct(ClassNameCheck $classCheck, bool $checkClassCaseSensitivity)
    {
        $this->classCheck = $classCheck;
        $this->checkClassCaseSensitivity = $checkClassCaseSensitivity;
    }
    /**
     * @param  array<RequireExtendsTag> $extendsTags
     * @return RuleError[]
     */
    public function checkExtendsTags(Node $node, array $extendsTags) : array
    {
        $errors = [];
        if (count($extendsTags) > 1) {
            $errors[] = RuleErrorBuilder::message(sprintf('PHPDoc tag @phpstan-require-extends can only be used once.'))->build();
        }
        foreach ($extendsTags as $extendsTag) {
            $type = $extendsTag->getType();
            if (!$type instanceof ObjectType) {
                $errors[] = RuleErrorBuilder::message(sprintf('PHPDoc tag @phpstan-require-extends contains non-object type %s.', $type->describe(VerbosityLevel::typeOnly())))->build();
                continue;
            }
            $class = $type->getClassName();
            $referencedClassReflection = $type->getClassReflection();
            if ($referencedClassReflection === null) {
                $errors[] = RuleErrorBuilder::message(sprintf('PHPDoc tag @phpstan-require-extends contains unknown class %s.', $class))->discoveringSymbolsTip()->build();
                continue;
            }
            if (!$referencedClassReflection->isClass()) {
                $errors[] = RuleErrorBuilder::message(sprintf('PHPDoc tag @phpstan-require-extends cannot contain non-class type %s.', $class))->build();
            } elseif ($referencedClassReflection->isFinal()) {
                $errors[] = RuleErrorBuilder::message(sprintf('PHPDoc tag @phpstan-require-extends cannot contain final class %s.', $class))->build();
            } else {
                $errors = array_merge($errors, $this->classCheck->checkClassNames([new ClassNameNodePair($class, $node)], $this->checkClassCaseSensitivity));
            }
        }
        return $errors;
    }
}
