<?php
/*
 * Your installation or use of this SugarCRM file is subject to the applicable
 * terms available at
 * http://support.sugarcrm.com/Resources/Master_Subscription_Agreements/.
 * If you do not agree to all of the applicable terms or do not have the
 * authority to bind the entity as an authorized representative, then do not
 * install or use this SugarCRM file.
 *
 * Copyright (C) SugarCRM Inc. All rights reserved.
 */
$dictionary['PurchasedLineItem'] = [
    'table' => 'purchased_line_items',
    'color' => 'army',
    'icon' => 'sicon-purchased-line-items-lg',
    'optimistic_locking' => true,
    'audited' => true,
    'activity_enabled' => true,
    'unified_search' => true,
    'full_text_search' => true,
    'comment' => 'The user (not Admin) view of a Purchased Item; an instance of a purchase.',
    'recalculations' => [
        [
            'field' => 'service_start_date',
        ],
        [
            'field' => 'service_end_date',
            'modifications' => [
                '+1 day',
            ],
        ],
    ],
    'fields' => [
        // Fields for this model
        'name' => [
            'name' => 'name',
            'vname' => 'LBL_NAME',
            'dbType' => 'varchar',
            'type' => 'name',
            'len' => '255',
            'unified_search' => true,
            'full_text_search' => [
                'enabled' => true,
                'searchable' => true,
            ],
            'comment' => 'Name of the product',
            'reportable' => true,
            'importable' => 'required',
            'required' => true,
            'audited' => true,
        ],
        'annual_revenue' => [
            'name' => 'annual_revenue',
            'vname' => 'LBL_ANNUAL_REVENUE',
            'type' => 'currency',
            'len' => '26,6',
            'calculated' => true,
            'enforced' => true,
            'formula' => 'ifElse(equal($service, "1"),
                            ifElse(equal($service_duration_unit, "year"),
                                currencyDivide($total_amount, $service_duration_value),
                                ifElse(equal($service_duration_unit, "month"),
                                    currencyMultiply(currencyDivide($total_amount, $service_duration_value), 12),
                                    ifElse(equal($service_duration_unit, "day"),
                                        currencyMultiply(currencyDivide($total_amount, $service_duration_value), 365),
                                        ""
                                    )
                                )
                            ),
                            $total_amount                            
                        )',
        ],
        'book_value_usdollar' => [
            'name' => 'book_value_usdollar',
            'vname' => 'LBL_BOOK_VALUE_USDOLLAR',
            'group' => 'book_value',
            'type' => 'currency',
            'len' => '26,6',
            'comment' => 'Book value expressed in USD',
            'readonly' => true,
            'is_base_currency' => true,
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'formula' => 'ifElse(isNumeric($book_value), currencyDivide($book_value, $base_rate), "")',
            'calculated' => true,
            'enforced' => true,
        ],
        'cost_price' => [
            'name' => 'cost_price',
            'vname' => 'LBL_COST_PRICE',
            'type' => 'currency',
            'len' => '26,6',
            'audited' => true,
            'comment' => 'Product cost',
        ],
        'cost_usdollar' => [
            'name' => 'cost_usdollar',
            'vname' => 'LBL_COST_USDOLLAR',
            'group' => 'cost_price',
            'type' => 'currency',
            'currency_id' => '-99',
            'len' => '26,6',
            'comment' => 'Cost expressed in USD',
            'readonly' => true,
            'is_base_currency' => true,
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'formula' => 'ifElse(isNumeric($cost_price), currencyDivide($cost_price, $base_rate), "")',
            'calculated' => true,
            'enforced' => true,
        ],
        'date_closed' => [
            'name' => 'date_closed',
            'vname' => 'LBL_DATE_CLOSED',
            'required' => true,
            'display_default' => 'now',
            'type' => 'date',
            'audited' => true,
            'comment' => 'Expected or actual date the purchased line item will close',
            'importable' => 'required',
            'enable_range_search' => true,
            'options' => 'date_range_search_dom',
            'related_fields' => [
                'date_closed_timestamp',
            ],
            'full_text_search' => [
                'enabled' => true,
                'searchable' => false,
            ],
        ],
        'date_closed_timestamp' => [
            'name' => 'date_closed_timestamp',
            'vname' => 'LBL_DATE_CLOSED_TIMESTAMP',
            'type' => 'ulong',
            'reportable' => false,
            'audited' => true,
            'activity_enabled' => false,
            'formula' => 'timestamp($date_closed)',
            'calculated' => true,
            'enforced' => true,
            'importable' => false,
        ],
        'deal_calc' => [
            'name' => 'deal_calc',
            'vname' => 'LBL_DEAL_CALC',
            'type' => 'currency',
            'len' => '26,6',
            'group' => 'deal_calc',
            'comment' => 'Calculated Revenue or Loss',
            'formula' => 'ifElse(equal($discount_select, "1"),
                            currencyMultiply(ifElse(isNumeric($subtotal), $subtotal, multiply($discount_price, $quantity)), currencyDivide($discount_amount, 100)),
                            ifElse(isNumeric($discount_amount_signed), $discount_amount_signed, 0)
                        )',
            'calculated' => true,
            'enforced' => true,
            'related_fields' => [
                'currency_id',
                'base_rate',
                'discount_price',
                'quantity',
                'discount_amount',
                'subtotal',
            ],
        ],
        'deal_calc_usdollar' => [
            'name' => 'deal_calc_usdollar',
            'vname' => 'LBL_DEAL_CALC_USDOLLAR',
            'type' => 'currency',
            'currency_id' => '-99',
            'len' => '26,6',
            'group' => 'deal_calc',
            'comment' => 'Calculated Revenue or Loss in US Dollars',
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'formula' => 'ifElse(isNumeric($deal_calc), currencyDivide($deal_calc, $base_rate), "")',
            'calculated' => true,
            'enforced' => true,
        ],
        'discount_amount' => [
            'name' => 'discount_amount',
            'vname' => 'LBL_TOTAL_DISCOUNT_AMOUNT',
            'dbType' => 'currency',
            'type' => 'discount-amount',
            'len' => '26,6',
            'default' => '0',
            'precision' => 6,
            'comment' => 'Discounted amount (Displayed as Total Discount Amount)',
            'studio' => false,
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'studio' => false,
        ],
        'discount_amount_signed' => [
            'name' => 'discount_amount_signed',
            'vname' => 'LBL_DISCOUNT_AMOUNT_SIGNED',
            'type' => 'currency',
            'len' => '26,6',
            'default' => '0',
            'audited' => true,
            'comment' => 'Discounted Amount Signed',
            'formula' => 'subtract(ifElse(isNumeric($subtotal), $subtotal, multiply($discount_price, $quantity)), $total_amount)',
            'calculated' => true,
            'studio' => false,
            'enforced' => true,
        ],
        'discount_amount_usdollar' => [
            'name' => 'discount_amount_usdollar',
            'vname' => 'LBL_DISCOUNT_RATE_USDOLLAR',
            'type' => 'decimal',
            'len' => '26,6',
            'readonly' => true,
            'is_base_currency' => true,
            'formula' => 'ifElse(isNumeric($discount_amount), currencyDivide($discount_amount, $base_rate), "")',
            'calculated' => true,
            'enforced' => true,
        ],
        'discount_price' => [
            'name' => 'discount_price',
            'vname' => 'LBL_DISCOUNT_PRICE',
            'type' => 'currency',
            'len' => '26,6',
            'audited' => true,
            'required' => true,
            'comment' => 'Discounted price ("Unit Price" in Quote)',
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'formula' => '
            ifElse(
                and(
                    equal($product_template_id, ""),
                    not(isNumeric($discount_price)),
                    greaterThan($quantity, 0)
                ),
                divide($revenue, $quantity),
                $discount_price
            )',
            'enforced' => false,
            'calculated' => true,
        ],
        'discount_select' => [
            'name' => 'discount_select',
            'vname' => 'LBL_SELECT_DISCOUNT',
            'type' => 'bool',
            'reportable' => false,
            'importable' => false,
            'studio' => false,
        ],
        'discount_usdollar' => [
            'name' => 'discount_usdollar',
            'vname' => 'LBL_DISCOUNT_USDOLLAR',
            'group' => 'discount_price',
            'type' => 'currency',
            'currency_id' => '-99',
            'len' => '26,6',
            'comment' => 'Discount price expressed in USD',
            'readonly' => true,
            'is_base_currency' => true,
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'formula' => 'ifElse(isNumeric($discount_price), currencyDivide($discount_price, $base_rate), "")',
            'calculated' => true,
            'enforced' => true,
        ],
        'list_usdollar' => [
            'name' => 'list_usdollar',
            'vname' => 'LBL_LIST_USDOLLAR',
            'type' => 'currency',
            'currency_id' => '-99',
            'group' => 'list_price',
            'len' => '26,6',
            'comment' => 'List price expressed in USD',
            'readonly' => true,
            'is_base_currency' => true,
            'related_fields' => [
                'currency_id',
                'base_rate',
            ],
            'formula' => 'ifElse(isNumeric($list_price), currencyDivide($list_price, $base_rate), "")',
            'calculated' => true,
            'enforced' => true,
        ],
        'mft_part_num' => [
            'name' => 'mft_part_num',
            'vname' => 'LBL_MFT_PART_NUM',
            'type' => 'varchar',
            'len' => '50',
            'comment' => 'Manufacturer part number',
        ],
        'revenue' => [
            'formula' => 'ifElse(equal($revenue,""),string($total_amount),$revenue)',
            'calculated' => true,
            'name' => 'revenue',
            'vname' => 'LBL_REVENUE',
            'required' => true,
            'type' => 'currency',
            'len' => '26,6',
            'audited' => true,
            'showTransactionalAmount' => true,
            'related_fields' => [
                'currency_id',
                'base_rate',
                'total_amount',
                'quantity',
                'discount_amount',
                'discount_price',
            ],
        ],
        'revenue_usdollar' => [
            'name' => 'revenue_usdollar',
            'vname' => 'LBL_REVENUE_USDOLLAR',
            'type' => 'currency',
            'currency_id' => '-99',
            'len' => '26,6',
            'comment' => 'Revenue expressed in USD',
        ],
        'total_amount' => [
            'name' => 'total_amount',
            'formula' => '
                ifElse(and(isNumeric($quantity), isNumeric($discount_price)),
                    currencySubtract(
                        ifElse(isNumeric($subtotal), $subtotal, multiply($discount_price, $quantity)),
                        ifElse(equal($discount_select, "1"),
                            currencyMultiply(ifElse(isNumeric($subtotal), $subtotal, multiply($discount_price, $quantity)), currencyDivide($discount_amount, 100)),
                            ifElse(greaterThan($quantity, 0), ifElse(isNumeric(toString($discount_amount)),
                                ifElse(greaterThan(0, $discount_price), negate($discount_amount), $discount_amount), 0),
                            ifElse(isNumeric(toString($discount_amount)), negate($discount_amount), 0))
                        )
                    ),
                    ""
                )
            ',
            'calculated' => true,
            'enforced' => true,
            'vname' => 'LBL_CALCULATED_LINE_ITEM_AMOUNT',
            'reportable' => false,
            'type' => 'currency',
            'related_fields' => [
                'discount_price',
                'quantity',
                'discount_amount',
                'discount_select',
            ],
        ],
        'subtotal' => [
            'name' => 'subtotal',
            'vname' => 'LBL_SUBTOTAL',
            'type' => 'currency',
            'len' => '26,6',
            'calculated' => true,
            'enforced' => true,
            'formula' => '
                ifElse(
                    and(isNumeric($discount_price), isNumeric($quantity)),
                    ifElse(
                        and(
                            isNumeric($service_duration_value),
                            isNumeric($catalog_service_duration_value),
                            not(equal($service_duration_unit, "")),
                            not(equal($catalog_service_duration_unit, ""))
                        ),
                        prorateValue(
                            multiply($discount_price, $quantity),
                            number($service_duration_value),
                            $service_duration_unit,
                            number($catalog_service_duration_value),
                            $catalog_service_duration_unit
                        ),
                        multiply($discount_price, $quantity)
                    ),
                    0
                )
            ',
            'comment' => 'The total of the line item before any discounts are applied, taking any proration into consideration',
            'related_fields' => [
                'discount_price',
                'quantity',
                'service_duration_value',
                'service_duration_unit',
                'catalog_service_duration_value',
                'catalog_service_duration_unit',
            ],
            'additionalCalculationTriggerFields' => [
                'discount_amount',
                'discount_select',
            ],
        ],
        // Fields for special relationships
        'categories' => [
            'name' => 'categories',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_categories',
            'vname' => 'LBL_PRODUCT_CATEGORIES',
            'link_type' => 'one',
            'module' => 'ProductCategories',
            'bean_name' => 'ProductCategory',
            'source' => 'non-db',
        ],
        'category_id' => [
            'name' => 'category_id',
            'vname' => 'LBL_CATEGORY_ID',
            'type' => 'id',
            'group' => 'category_name',
            'required' => false,
            'reportable' => true,
            'comment' => 'Product category',
        ],
        'category_name' => [
            'name' => 'category_name',
            'rname' => 'name',
            'readonly' => true,
            'id_name' => 'category_id',
            'vname' => 'LBL_CATEGORY_NAME',
            'join_name' => 'categories',
            'type' => 'relate',
            'link' => 'categories',
            'table' => 'product_categories',
            'isnull' => 'true',
            'module' => 'ProductCategories',
            'dbType' => 'varchar',
            'len' => '255',
            'save' => true,
            'source' => 'non-db',
            'required' => false,
        ],
        'manufacturer' => [
            'name' => 'manufacturer',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_manufacturers',
            'vname' => 'LBL_MANUFACTURER',
            'link_type' => 'one',
            'module' => 'Manufacturers',
            'bean_name' => 'Manufacturer',
            'source' => 'non-db',
        ],
        'manufacturer_id' => [
            'name' => 'manufacturer_id',
            'vname' => 'LBL_MANUFACTURER',
            'type' => 'id',
            'required' => false,
            'reportable' => false,
            'massupdate' => false,
            'comment' => 'Manufacturer of product',
        ],
        'manufacturer_name' => [
            'name' => 'manufacturer_name',
            'rname' => 'name',
            'id_name' => 'manufacturer_id',
            'type' => 'relate',
            'vname' => 'LBL_MANUFACTURER_NAME',
            'join_name' => 'manufacturers',
            'link' => 'manufacturer',
            'table' => 'manufacturers',
            'isnull' => 'true',
            'source' => 'non-db',
            'module' => 'Manufacturers',
            'dbType' => 'varchar',
            'len' => '255',
            'massupdate' => false,
            'related_fields' => [
                'manufacturer_id',
            ],
        ],
        'product_templates' => [
            'name' => 'product_templates',
            'type' => 'link',
            'relationship' => 'purhcasedlineitem_templates',
            'vname' => 'LBL_PRODUCT_TEMPLATES',
            'link_type' => 'one',
            'module' => 'ProductTemplates',
            'bean_name' => 'ProductTemplate',
            'source' => 'non-db',
        ],
        'product_template_id' => [
            'name' => 'product_template_id',
            'type' => 'id',
            'vname' => 'LBL_PRODUCT_TEMPLATE_ID',
            'required' => false,
            'reportable' => false,
            'comment' => 'Product (in Admin Products) from which this purchased item is derived.',
        ],
        'product_template_name' => [
            'name' => 'product_template_name',
            'rname' => 'name',
            'readonly' => true,
            'id_name' => 'product_template_id',
            'vname' => 'LBL_PRODUCT',
            'join_name' => 'templates',
            'type' => 'relate',
            'save' => true,
            'link' => 'product_templates',
            'table' => 'product_templates',
            'isnull' => 'true',
            'module' => 'ProductTemplates',
            'dbType' => 'varchar',
            'len' => '255',
            'source' => 'non-db',
            'auto_populate' => true,
            'populate_list' => [
                'name' => 'name',
                'mft_part_num' => 'mft_part_num',
                'list_price' => 'list_price',
                'cost_price' => 'cost_price',
                'discount_price' => 'discount_price',
                'list_usdollar' => 'list_usdollar',
                'cost_usdollar' => 'cost_usdollar',
                'discount_usdollar' => 'discount_usdollar',
                'tax_class' => 'tax_class',
                'weight' => 'weight',
                'manufacturer_id' => 'manufacturer_id',
                'manufacturer_name' => 'manufacturer_name',
                'currency_id' => 'currency_id',
                'base_rate' => 'base_rate',
                'renewable' => 'renewable',
                'service_duration_value' => ['service_duration_value', 'catalog_service_duration_value'],
                'service_duration_unit' => ['service_duration_unit', 'catalog_service_duration_unit'],
            ],
        ],
        'lock_duration' => [
            'name' => 'lock_duration',
            'vname' => 'LBL_LOCK_DURATION',
            'type' => 'bool',
            'default' => 0,
            'studio' => false,
            'enforced' => true,
            'calculated' => true,
            'massupdate' => false,
            'formula' => 'related($product_templates, "lock_duration")',
            'comment' => 'Turn on or off a user ability to directly edit the duration field',
        ],
        'product_type' => [
            'name' => 'product_type',
            'type' => 'link',
            'relationship' => 'purhcasedlineitem_product_types',
            'vname' => 'LBL_PRODUCT_TYPES',
            'link_type' => 'one',
            'module' => 'ProductTypes',
            'bean_name' => 'ProductType',
            'source' => 'non-db',
        ],
        'product_type_id' => [
            'name' => 'product_type_id',
            'vname' => 'LBL_TYPE',
            'type' => 'id',
            'required' => false,
            'reportable' => false,
            'comment' => 'Product type (ex: hardware, software)',
        ],
        'product_type_name' => [
            'name' => 'product_type_name',
            'rname' => 'name',
            'readonly' => true,
            'id_name' => 'product_type_id',
            'vname' => 'LBL_PRODUCT_TYPE',
            'join_name' => 'types',
            'type' => 'relate',
            'save' => true,
            'link' => 'product_type',
            'table' => 'product_types',
            'isnull' => 'true',
            'module' => 'ProductTypes',
            'importable' => 'false',
            'dbType' => 'varchar',
            'len' => '255',
            'source' => 'non-db',
        ],
        'purchase' => [
            'name' => 'purchase',
            'type' => 'link',
            'relationship' => 'purchase_purchasedlineitems',
            'vname' => 'LBL_PURCHASE',
            'link_type' => 'one',
            'side' => 'right',
            'source' => 'non-db',
        ],
        'purchase_id' => [
            'name' => 'purchase_id',
            'vname' => 'LBL_PURCHASE',
            'type' => 'id',
            'reportable' => false,
            'comment' => 'Purchase which spawned this PLI',
        ],
        'purchase_name' => [
            'name' => 'purchase_name',
            'rname' => 'name',
            'id_name' => 'purchase_id',
            'vname' => 'LBL_PURCHASE',
            'related_fields' => [
                'purchase_id',
            ],
            'join_name' => 'purchase',
            'type' => 'relate',
            'save' => true,
            'link' => 'purchase',
            'table' => 'purchases',
            'isnull' => 'true',
            'module' => 'Purchases',
            'dbType' => 'varchar',
            'len' => '255',
            'source' => 'non-db',
            'unified_search' => true,
            'importable' => true,
            'exportable' => true,
            'required' => true,
            'auto_populate' => true,
            'populate_list' => [
                'type_name' => 'product_type_name',
                'type_id' => 'product_type_id',
                'category_id' => 'category_id',
                'category_name' => 'category_name',
                'product_template_name' => 'product_template_name',
                'product_template_id' => 'product_template_id',
                'service' => 'service',
                'account_id' => 'account_id',
                'account_name' => 'account_name',
            ],
        ],
        'revenuelineitem' => [
            'name' => 'revenuelineitem',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_revenuelineitem',
            'vname' => 'LBL_RLI',
            'source' => 'non-db',
        ],
        'revenuelineitem_id' => [
            'name' => 'revenuelineitem_id',
            'type' => 'id',
            'vname' => 'LBL_RLI_ID',
            'required' => false,
            'reportable' => false,
            'comment' => 'If PLI was created from RLI, this is that RLIs ID',
        ],
        'revenuelineitem_name' => [
            'name' => 'revenuelineitem_name',
            'rname' => 'name',
            'id_name' => 'revenuelineitem_id',
            'join_name' => 'revenuelineitem',
            'type' => 'relate',
            'link' => 'revenuelineitem',
            'table' => 'revenue_line_items',
            'isnull' => true,
            'required' => false,
            'module' => 'RevenueLineItems',
            'dbType' => 'varchar',
            'len' => '255',
            'vname' => 'LBL_RLI_NAME',
            'source' => 'non-db',
            'comment' => 'RLI Name',
            'showInMode' => 'RevenueLineItems',
        ],
        // Fields for "Activity" relationships
        'calls' => [
            'name' => 'calls',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_calls',
            'module' => 'Calls',
            'bean_name' => 'Call',
            'source' => 'non-db',
            'vname' => 'LBL_CALLS',
        ],
        'documents' => [
            'name' => 'documents',
            'type' => 'link',
            'relationship' => 'documents_purchasedlineitems',
            'source' => 'non-db',
            'vname' => 'LBL_DOCUMENTS_SUBPANEL_TITLE',
        ],
        'emails' => [
            'name' => 'emails',
            'type' => 'link',
            'relationship' => 'emails_purchasedlineitems_rel', /* reldef in emails */
            'module' => 'Emails',
            'bean_name' => 'Email',
            'source' => 'non-db',
            'vname' => 'LBL_EMAILS',
        ],
        'meetings' => [
            'name' => 'meetings',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_meetings',
            'module' => 'Meetings',
            'bean_name' => 'Meeting',
            'source' => 'non-db',
            'vname' => 'LBL_MEETINGS',
        ],
        'notes' => [
            'name' => 'notes',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_notes',
            'source' => 'non-db',
            'vname' => 'LBL_NOTES',
        ],
        'tasks' => [
            'name' => 'tasks',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_tasks',
            'source' => 'non-db',
            'vname' => 'LBL_NOTES',
        ],
        'addon_qlis' => [
            'name' => 'addon_qlis',
            'type' => 'link',
            'relationship' => 'product_pli_addons',
            'vname' => 'LBL_QLI_ADDONS',
            'module' => 'Products',
            'bean_name' => 'Product',
            'source' => 'non-db',
            'workflow' => false,
        ],
        'addon_rlis' => [
            'name' => 'addon_rlis',
            'type' => 'link',
            'relationship' => 'revenuelineitem_pli_addons',
            'vname' => 'LBL_RLI_ADDONS',
            'module' => 'RevenueLineItems',
            'bean_name' => 'RevenueLineItem',
            'source' => 'non-db',
            'workflow' => false,
        ],
        'renewal_opp_link' => [
            'name' => 'renewal_opp_link',
            'type' => 'link',
            'relationship' => 'purchasedlineitem_renewal_opp',
            'vname' => 'LBL_RENEWAL_OPPORTUNITY',
            'link_type' => 'one',
            'module' => 'Opportunities',
            'bean_name' => 'Opportunity',
            'source' => 'non-db',
        ],
        'renewal_opp_id' => [
            'name' => 'renewal_opp_id',
            'type' => 'id',
            'vname' => 'LBL_RENEWAL_OPPORTUNITY_ID',
            'required' => false,
            'reportable' => false,
            'comment' => 'Renewal opportunity of this PLI',
        ],
        'renewal_opp_name' => [
            'name' => 'renewal_opp_name',
            'rname' => 'name',
            'id_name' => 'renewal_opp_id',
            'save' => true,
            'vname' => 'LBL_RENEWAL_OPPORTUNITY',
            'join_name' => 'opportunities',
            'type' => 'relate',
            'link' => 'renewal_opp_link',
            'table' => 'opportunities',
            'module' => 'Opportunities',
            'source' => 'non-db',
            'massupdate' => false,
        ],
        'account_link' => [
            'name' => 'account_link',
            'type' => 'link',
            'relationship' => 'purchasedlineitems_accounts',
            'vname' => 'LBL_ACCOUNT',
            'link_type' => 'one',
            'module' => 'Accounts',
            'bean_name' => 'Account',
            'source' => 'non-db',
        ],
        'account_id' => [
            'name' => 'account_id',
            'type' => 'id',
            'vname' => 'LBL_ACCOUNT_ID',
            'required' => false,
            'reportable' => false,
            'audited' => true,
            'comment' => 'Account related to the parent Purchase of this PLI',
            'formula' => 'ifElse(related($purchase, "account_id"), related($purchase, "account_id"), $account_id)',
            'enforced' => true,
            'calculated' => true,
        ],
        'account_name' => [
            'name' => 'account_name',
            'rname' => 'name',
            'id_name' => 'account_id',
            'save' => true,
            'vname' => 'LBL_ACCOUNT',
            'join_name' => 'accounts',
            'type' => 'relate',
            'link' => 'account_link',
            'table' => 'accounts',
            'module' => 'Accounts',
            'source' => 'non-db',
            'massupdate' => false,
        ],
        'status' => [
            'name' => 'status',
            'vname' => 'LBL_STATUS',
            'type' => 'enum',
            'options' => 'purchased_line_items_status_dom',
            'calculated' => true,
            'enforced' => true,
            'formula' => '
                ifElse(
                    and(
                        not(equal($service_start_date, "")),
                        not(equal($service_end_date, ""))
                    ),
                    ifElse(
                        greaterThan(timestamp($service_start_date), timestamp(today())),
                        "Future",
                        ifElse(
                            and(
                                equal($service, 1),
                                greaterThan(timestamp(addDays($service_end_date, 1)), timestamp(today()))
                            ),
                            "Active",
                            "Past"
                        )
                    ),
                    ""
                )
            ',
            'massupdate' => false,
            'reportable' => true,
            'importable' => 'false',
            'duplicate_merge' => 'disabled',
            'hidemassupdate' => true,
            'studio' => [
                'field' => [
                    'options' => false,
                ],
                'default' => false,
                'calculated' => false,
                'importable' => false,
                'duplicate_merge' => false,
            ],
        ],
    ],
    'indices' => [
        [
            'name' => 'idx_purchaseid_del',
            'type' => 'index',
            'fields' => [
                'purchase_id',
                'deleted',
            ],
        ],
        [
            'name' => 'idx_pli_purchaseid_del',
            'type' => 'index',
            'fields' => [
                'deleted',
                'purchase_id',
                'id',
            ],
        ],
        [
            'name' => 'idx_pli_dclose_del',
            'type' => 'index',
            'fields' => [
                'deleted',
                'date_closed',
                'id',
            ],
        ],
        [
            'name' => 'idx_pli_dent_name_del',
            'type' => 'index',
            'fields' => [
                'deleted',
                'date_entered',
                'name',
                'id',
            ],
        ],
        [
            'name' => 'idx_pli_dclose_name_del',
            'type' => 'index',
            'fields' => [
                'deleted',
                'date_closed',
                'name',
                'id',
            ],
        ],
        [
            'name' => 'idx_pli_catid_del',
            'type' => 'index',
            'fields' => [
                'deleted',
                'category_id',
                'id',
            ],
        ],
        [
            'name' => 'idx_pli_manid_del',
            'type' => 'index',
            'fields' => [
                'deleted',
                'manufacturer_id',
                'id',
            ],
        ],
        [
            'name' => 'idx_pli_del_accountid',
            'type' => 'index',
            'fields' => [
                'deleted',
                'account_id',
            ],
        ],
        [
            'name' => 'idx_pli_del_revenuelineitemid',
            'type' => 'index',
            'fields' => [
                'deleted',
                'revenuelineitem_id',
            ],
        ],
    ],
    'relationships' => [
        'purchasedlineitem_categories' => [
            'lhs_module' => 'ProductCategories',
            'lhs_table' => 'product_categories',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'category_id',
            'relationship_type' => 'one-to-many',
        ],
        'purchasedlineitem_manufacturers' => [
            'lhs_module' => 'Manufacturers',
            'lhs_table' => 'manufacturers',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'manufacturer_id',
            'relationship_type' => 'one-to-many',
        ],
        'purhcasedlineitems_modified_user' => [
            'lhs_module' => 'Users',
            'lhs_table' => 'users',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'modified_user_id',
            'relationship_type' => 'one-to-many',
        ],
        'purhcasedlineitem_product_types' => [
            'lhs_module' => 'ProductTypes',
            'lhs_table' => 'product_types',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'type_id',
            'relationship_type' => 'one-to-many',
        ],
        'purchasedlineitem_revenuelineitem' => [
            'lhs_module' => 'RevenueLineItems',
            'lhs_table' => 'revenue_line_items',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'revenuelineitem_id',
            'relationship_type' => 'one-to-one',
        ],
        'purhcasedlineitem_templates' => [
            'lhs_module' => 'ProductTemplates',
            'lhs_table' => 'product_templates',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'product_template_id',
            'relationship_type' => 'one-to-many',
        ],
        'purchasedlineitem_calls' => [
            'lhs_module' => 'PurchasedLineItems',
            'lhs_table' => 'purchased_line_items',
            'lhs_key' => 'id',
            'rhs_module' => 'Calls',
            'rhs_table' => 'calls',
            'rhs_key' => 'parent_id',
            'relationship_type' => 'one-to-many',
            'relationship_role_column' => 'parent_type',
            'relationship_role_column_value' => 'PurchasedLineItems',
        ],
        'purchasedlineitem_meetings' => [
            'lhs_module' => 'PurchasedLineItems',
            'lhs_table' => 'purchased_line_items',
            'lhs_key' => 'id',
            'rhs_module' => 'Meetings',
            'rhs_table' => 'meetings',
            'rhs_key' => 'parent_id',
            'relationship_type' => 'one-to-many',
            'relationship_role_column' => 'parent_type',
            'relationship_role_column_value' => 'PurchasedLineItems',
        ],
        'purchasedlineitem_notes' => [
            'lhs_module' => 'PurchasedLineItems',
            'lhs_table' => 'purchased_line_items',
            'lhs_key' => 'id',
            'rhs_module' => 'Notes',
            'rhs_table' => 'notes',
            'rhs_key' => 'parent_id',
            'relationship_type' => 'one-to-many',
            'relationship_role_column' => 'parent_type',
            'relationship_role_column_value' => 'PurchasedLineItems',
        ],
        'purchasedlineitem_tasks' => [
            'lhs_module' => 'PurchasedLineItems',
            'lhs_table' => 'purchased_line_items',
            'lhs_key' => 'id',
            'rhs_module' => 'Tasks',
            'rhs_table' => 'tasks',
            'rhs_key' => 'parent_id',
            'relationship_type' => 'one-to-many',
            'relationship_role_column' => 'parent_type',
            'relationship_role_column_value' => 'PurchasedLineItems',
        ],
        'purchasedlineitem_renewal_opp' => [
            'lhs_module' => 'Opportunities',
            'lhs_table' => 'opportunities',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'renewal_opp_id',
            'relationship_type' => 'one-to-many',
        ],
        'purchasedlineitems_accounts' => [
            'lhs_module' => 'Accounts',
            'lhs_table' => 'accounts',
            'lhs_key' => 'id',
            'rhs_module' => 'PurchasedLineItems',
            'rhs_table' => 'purchased_line_items',
            'rhs_key' => 'account_id',
            'relationship_type' => 'one-to-many',
        ],
    ],
];

VarDefManager::createVardef(
    'PurchasedLineItems',
    'PurchasedLineItem',
    [
        'basic',
        'assignable',
        'team_security',
        'sales_item',
        'audit',
    ]
);
