<?php

/*
 * Your installation or use of this SugarCRM file is subject to the applicable
 * terms available at
 * http://support.sugarcrm.com/Resources/Master_Subscription_Agreements/.
 * If you do not agree to all of the applicable terms or do not have the
 * authority to bind the entity as an authorized representative, then do not
 * install or use this SugarCRM file.
 *
 * Copyright (C) SugarCRM Inc. All rights reserved.
 */

class ConvertLayoutMetadataParser extends GridLayoutMetaDataParser
{
    /**
     * @var \SugarBean|null|mixed
     */
    public $seed;
    /**
     * @var \History|mixed
     */
    //@codingStandardsIgnoreStart
    public $_history;
    //@codingStandardsIgnoreEnd
    protected $pathMap = [
        MB_BASEMETADATALOCATION => '',
        MB_CUSTOMMETADATALOCATION => 'custom/',
        MB_WORKINGMETADATALOCATION => 'custom/working/',
        MB_HISTORYMETADATALOCATION => 'custom/history/',
    ];
    protected $fileName = 'modules/Leads/clients/base/layouts/convert-main/convert-main.php';
    protected $_convertdefs; //lead convert metadata pulled out for convenience
    protected $defaultModuleDefSettings = [
        'required' => false,
        'copyData' => false,
        'duplicateCheckOnStart' => false,
    ];

    protected $excludedModules = [
        'Activities',
        'Products',
        'ProductTemplates',
        'ProductBundles',
        'Leads',
        'Users',
        'pmse_Business_Rules',
        'pmse_Emails_Templates',
        'pmse_Inbox',
        'pmse_Project',
        'RevenueLineItems',
    ];

    //fields that should be hidden in the create layout
    protected $excludedFields = [
        'Calls' => [
            'repeat_type' => 'Calls',
        ],
        'Meetings' => [
            'repeat_type' => 'Meetings',
        ],
    ];

    public function __construct($module)
    {
        $this->FILLER = [
            'name' => MBConstants::$FILLER['name'],
            'label' => translate(MBConstants::$FILLER['label']),
        ];
        $this->seed = BeanFactory::newBean($module);
        $this->_moduleName = $module;
        $this->_view = MB_EDITVIEW;
        $this->_fielddefs = $this->seed->field_defs;
        $this->loadViewDefs();
        $this->_history = new History($this->pathMap[MB_HISTORYMETADATALOCATION] . $this->fileName);
    }

    public function getOriginalViewDefs()
    {
        $viewdefs = [];
        //load from the original file only
        include $this->fileName;
        return $viewdefs;
    }

    public function getLanguage()
    {
        return '';
    }

    public function getHistory()
    {
        return $this->_history;
    }

    /**
     * Override parent and noop - we don't use this for convert lead
     *
     * {@inheritDoc}
     */
    public function handleSave($populate = true, $clearCache = true)
    {
    }

    /**
     * Merge new data with existing data and deploy the resulting convert def
     *
     * @param array $data
     */
    public function updateConvertDef($data)
    {
        $isUsingRLIsInConvertBefore = Lead::isUsingRLIsInConvert();
        $this->_convertdefs['modules'] = $this->mergeConvertDefs($data);
        $this->deploy();
        $isUsingRLIsInConvertAfter = Lead::isUsingRLIsInConvert();
        if ($isUsingRLIsInConvertBefore !== $isUsingRLIsInConvertAfter) {
            $leadViews = new LeadViews();
            $leadViews->toggleConvertDashboardProductDashlets($isUsingRLIsInConvertAfter);
        }
    }

    /**
     * Should take in an updated set of definitions and override the current panel set with the new one.
     *
     * @param $data
     * @return array result of merging the new data with existing data
     */
    public function mergeConvertDefs($data, $includeDefaults = false)
    {
        $includedModules = [];
        foreach ($data as $newDef) {
            if (!empty($newDef['module'])) {
                $includedModules[] = $newDef['module'];
            }
        }

        //Create the new convertdefs, replacing any properties in the modules with the ones from the request
        $final = [];
        foreach ($data as $newDef) {
            if (empty($newDef['module'])) {
                continue;
            }
            if ($includeDefaults) {
                $existingDef = $this->getDefaultDefForModule($newDef['module']);
            } else {
                $existingDef = $this->getDefForModule($newDef['module']);
            }
            if ($existingDef) {
                foreach ($existingDef as $key => $item) {
                    if (!isset($newDef[$key])) {
                        $newDef[$key] = $item;
                    }
                }
            }
            //if Opp is in the list, Account must be set to required
            if ($newDef['module'] === 'Accounts' && in_array('Opportunities', $includedModules)) {
                $newDef['required'] = true;
            }
            $newDef = $this->applyDependenciesAndHiddenFields($newDef, $includedModules);

            $final[] = $newDef;
        }
        return $final;
    }

    /**
     * Pull default def for module, check if dependencies & hidden fields apply
     * Add them if they do apply and remove the key altogether if they don't
     *
     * @param $def
     * @param $includedModules
     * @return mixed
     */
    protected function applyDependenciesAndHiddenFields($def, $includedModules)
    {
        $defaultDef = $this->getDefaultDefForModule($def['module']);

        if (isset($defaultDef['dependentModules'])) {
            $dependentModules = [];
            foreach ($defaultDef['dependentModules'] as $module => $value) {
                if (safeInArray($module, $includedModules)) {
                    $dependentModules[$module] = $value;
                }
            }
            if (!empty($dependentModules)) {
                $def['dependentModules'] = $dependentModules;
            } else {
                unset($def['dependentModules']);
            }
        }

        $hiddenFields = $this->getHiddenFields($def['module'], $defaultDef, $includedModules);
        if (!empty($hiddenFields)) {
            $def['hiddenFields'] = $hiddenFields;
        } else {
            unset($def['hiddenFields']);
        }

        return $def;
    }

    /**
     * Remove given module from the viewdefs module list
     *
     * @param string $module
     */
    public function removeLayout($module)
    {
        $moduleDefs = [];
        $newModuleDefs = [];
        if (isset($this->_convertdefs['modules'])) {
            $moduleDefs = $this->_convertdefs['modules'];
        }

        foreach ($moduleDefs as $moduleDef) {
            if ($moduleDef['module'] !== $module) {
                $newModuleDefs[] = $moduleDef;
            }
        }

        $this->_convertdefs['modules'] = $newModuleDefs;
        $this->deploy();
    }

    /**
     * Deploy the convert defs
     */
    public function deploy()
    {
        // when we deploy get rid of the working file; we have the changes in the MB_CUSTOMMETADATALOCATION so no need for a redundant copy in MB_WORKINGMETADATALOCATION
        // this also simplifies manual editing of layouts. You can now switch back and forth between Studio and manual changes without having to keep these two locations in sync
        $workingFilename = $this->pathMap[MB_WORKINGMETADATALOCATION] . $this->fileName;
        if (file_exists($workingFilename)) {
            unlink($workingFilename);
        }

        $filename = $this->pathMap[MB_CUSTOMMETADATALOCATION] . $this->fileName;
        $GLOBALS['log']->debug(get_class($this) . '->deploy(): writing to ' . $filename);
        $this->setConvertDef($this->_convertdefs);
        $this->_saveToFile($filename, $this->_viewdefs);
    }

    /**
     * Save the viewdefs to a file
     *
     * @param string $filename
     * @param array $defs
     */
    protected function _saveToFile($filename, $defs)
    {
        mkdir_recursive(dirname($filename));
        // create the new metadata file contents, and write it out
        if (!write_array_to_file('viewdefs', $defs, $filename)) {
            $GLOBALS ['log']->fatal(get_class($this) . ': could not write new viewdef file ' . $filename);
        }
    }

    /**
     * Load convert lead viewdefs from custom file if exists, base file otherwise
     */
    protected function loadViewDefs()
    {
        $viewdefs = [];
        $viewDefFile = SugarAutoLoader::existingCustomOne($this->fileName);
        include $viewDefFile;
        $this->_viewdefs = $viewdefs;
        $this->_convertdefs = $this->getConvertDef($this->_viewdefs);
    }

    public function getDefForModules()
    {
        return $this->_convertdefs['modules'];
    }

    /**
     * Convert metadata contains array of modules, retrieve a specific module def
     *
     * @param string $module module name
     * @param null $convertDefs
     * @return bool|array
     */
    public function getDefForModule($module, $convertDefs = null)
    {
        if (is_null($convertDefs)) {
            $convertDefs = $this->_convertdefs;
        }
        $moduleDef = false;
        foreach ($convertDefs['modules'] as $def) {
            if ($def['module'] === $module) {
                $moduleDef = $def;
            }
        }

        return $moduleDef;
    }

    /**
     * Set the convert defs for a given module
     * @param $module
     * @param $newDefs
     */
    public function setDefForModule($module, $newDefs)
    {
        // If the module already exists, update the existing defs
        foreach ($this->_convertdefs['modules'] as $idx => $existingDef) {
            if ($existingDef['module'] === $module) {
                $this->_convertdefs['modules'][$idx] = $newDefs;
                return;
            }
        }

        // Otherwise, add the new module at the end
        $this->_convertdefs['modules'][] = $newDefs;
    }

    /**
     * Retrieve the default definition for a module
     * If exists in original convert def, use that, otherwise, use the default def
     *
     * @param $module
     * @return array
     */
    public function getDefaultDefForModule($module)
    {
        $viewdefs = [];
        // check if custom def exists
        $fileName = "modules/$module/clients/base/layouts/convert-main-for-leads/convert-main-for-leads.php";

        if (!check_file_name($fileName)) {
            throw new Exception('The file path is incorrect');
        }

        if (file_exists("custom/$fileName")) {
            include "custom/$fileName";
        } elseif (file_exists($fileName)) {
            include $fileName;
        }

        if (isset($viewdefs[$module]['base']['layout']['convert-main-for-leads'])) {
            return $viewdefs[$module]['base']['layout']['convert-main-for-leads'];
        }

        // check if original def exists
        $originalViewDef = $this->getOriginalViewDefs();
        $originalConvertDef = $this->getConvertDef($originalViewDef);

        if ($moduleDef = $this->getDefForModule($module, $originalConvertDef)) {
            return $moduleDef;
        }

        // use default def
        $defaultModuleDef = array_merge(['module' => $module], $this->defaultModuleDefSettings);

        // if duplicate check is enabled for a module that is not already in the original viewdef
        // set the module to run duplicate check on start
        if ($this->isDupeCheckEnabledForModule($module)) {
            $defaultModuleDef['duplicateCheckOnStart'] = true;
        }

        return $defaultModuleDef;
    }

    /**
     * Check the vardef to determine if duplicate check is enabled for this module
     *
     * @param $module
     * @return bool
     */
    protected function isDupeCheckEnabledForModule($module)
    {
        global $beanList, $dictionary;

        $beanName = $beanList[$module] ?? '';
        return (isset($dictionary[$beanName]) &&
            isset($dictionary[$beanName]['duplicate_check']) &&
            isset($dictionary[$beanName]['duplicate_check']['enabled']) &&
            ($dictionary[$beanName]['duplicate_check']['enabled'] === true)
        );
    }

    /**
     * Get whether to use tabs in create form (we don't use tabs on lead convert)
     *
     * @return bool
     */
    public function getUseTabs()
    {
        return false;
    }

    /**
     * Set whether to use tabs in create form (leave false status - we don't use tabs on lead convert)
     *
     * @param bool $useTabs
     */
    public function setUseTabs($useTabs)
    {
    }

    /**
     * Retrieve the convert def from the viewdefs
     *
     * @param array $viewdefs
     * @return array
     */
    protected function getConvertDef($viewdefs)
    {
        if (isset($viewdefs['Leads']['base']['layout']['convert-main'])) {
            return $viewdefs['Leads']['base']['layout']['convert-main'];
        } else {
            return [];
        }
    }

    /**
     * Set the convert def at the appropriate location on the viewdef
     *
     * @param array $convertDef
     */
    protected function setConvertDef($convertDef)
    {
        $this->_viewdefs['Leads']['base']['layout']['convert-main'] = $convertDef;
    }

    /**
     * Check if module is allowed to be in convert flow
     *
     * @param $module
     * @return bool
     */
    public function isModuleAllowedInConvert($module)
    {
        //exclude modules that are in BWC or in the exclude list
        return (!isModuleBWC($module) && !in_array($module, $this->excludedModules));
    }

    /**
     * @param $module string Module for processing the hidden fields
     * @param $defaultDef array The default definitionns for the module
     * @param $includedModules array List of modules available
     *
     * @return array array of hidden fields for the module
     */
    public function getHiddenFields($module, $defaultDef, $includedModules)
    {
        $hiddenFields = [];

        $excludeFields = $this->getExcludedFields($module);
        $hiddenFieldsDef = $defaultDef['hiddenFields'] ?? [];
        $hiddenFieldsDef = array_merge($hiddenFieldsDef, $excludeFields);

        foreach ($hiddenFieldsDef as $fieldName => $module) {
            if (safeInArray($module, $includedModules)) {
                $hiddenFields[$fieldName] = $module;
            }
        }

        return $hiddenFields;
    }

    /**
     * Returns a list of fields that should be excluded from the layout
     * @param $module
     *
     * @return array
     */
    public function getExcludedFields($module)
    {
        return $this->excludedFields[$module] ?? [];
    }
}
