<?php
/*
 * Your installation or use of this SugarCRM file is subject to the applicable
 * terms available at
 * http://support.sugarcrm.com/Resources/Master_Subscription_Agreements/.
 * If you do not agree to all of the applicable terms or do not have the
 * authority to bind the entity as an authorized representative, then do not
 * install or use this SugarCRM file.
 *
 * Copyright (C) SugarCRM Inc. All rights reserved.
 */

use Doctrine\DBAL\Exception as DBALException;
use Sugarcrm\Sugarcrm\Util\Uuid;

require_once 'vendor/ytree/Tree.php';
require_once 'vendor/ytree/ExtNode.php';

/**
 * Polymorphic buckets - place any item in a folder
 */
class SugarFolder
{
    // public attributes
    public $id;
    public $name;
    public $parent_folder;
    public $has_child = 0; // flag node has child
    public $is_group = 0;
    public $is_dynamic = 0;
    public $dynamic_query = '';
    public $assign_to_id;
    public $created_by;
    public $modified_by;
    public $date_created;
    public $date_modified;
    public $deleted;
    public $folder_type;
    public $team_id;
    public $team_set_id;

    public $db;
    public $new_with_id = false;

    // core queries
    public $core = 'SELECT f.id, f.name, f.has_child, f.is_group, f.is_dynamic, f.dynamic_query, f.folder_type, f.created_by, i.deleted FROM folders f left join inbound_email i on f.id = i.groupfolder_id ';
    public $coreSubscribed = 'SELECT f.id, f.name, f.has_child, f.is_group, f.is_dynamic, f.dynamic_query, f.folder_type, f.created_by, i.deleted FROM folders f LEFT JOIN folders_subscriptions fs ON f.id = fs.folder_id LEFT JOIN inbound_email i on  i.groupfolder_id = f.id ';
    public $coreWhere = 'WHERE f.deleted = 0 ';
    public $coreWhereSubscribed = 'WHERE f.deleted = 0 AND fs.assigned_user_id = ';
    public $coreOrderBy = ' ORDER BY f.is_dynamic, f.is_group, f.name ASC ';

    public $hrSortLocal = [
        'flagged' => 'type',
        'status' => 'reply_to_status',
        'from' => 'emails_text.from_addr',
        'subject' => 'name',
        'date' => 'date_sent',
        'AssignedTo' => 'assigned_user_id',
        'flagged' => 'flagged',
    ];
    public $defaultSort = 'date';
    public $defaultDirection = 'DESC';

    protected $emailBean;

    // private attributes
    // @codingStandardsIgnoreLine PSR2.Classes.PropertyDeclaration.Underscore
    public $_depth;

    /**
     * folder field definition
     * @var array
     */
    private $fields = [];

    /**
     * folder subscription field definition
     * @var array
     */
    protected $folderSubscriptionFields = [];

    /**
     * folder table name
     * @var string
     */
    protected $table;

    /**
     * folder subscription table name
     * @var string
     */
    protected $folderSubscriptionTable;

    /** @var string */
    public $acl_team_set_id;

    /**
     * Sole constructor
     */
    public function __construct()
    {
        $this->db = DBManagerFactory::getInstance();
        $this->emailBean = BeanFactory::newBean('Emails');
        $dictionary = [];
        require 'metadata/foldersMetaData.php';
        $this->table = $dictionary['folders']['table'];
        $this->fields = $dictionary['folders']['fields'];
        $this->folderSubscriptionTable = $dictionary['folders_subscriptions']['table'];
        $this->folderSubscriptionFields = $dictionary['folders_subscriptions']['fields'];
    }

    public function deleteEmailFromAllFolder($id)
    {
        $this->db->getConnection()->delete(
            'folders_rel',
            ['polymorphic_module' => 'Emails', 'polymorphic_id' => $id]
        );
    }

    public function deleteEmailFromFolder($id)
    {
        $this->db->getConnection()->delete(
            'folders_rel',
            ['polymorphic_module' => 'Emails', 'polymorphic_id' => $id, 'folder_id' => $this->id]
        );
    }

    /**
     * @param string $id
     * @return bool
     * @throws DBALException
     */
    public function checkEmailExistForFolder($id)
    {
        $connection = $this->db->getConnection();
        $platform = $connection->getDatabasePlatform();
        $query = $platform->modifyLimitQuery(
            'SELECT NULL FROM folders_rel'
            . ' WHERE polymorphic_module = ? AND polymorphic_id = ? AND folder_id = ?',
            1
        );
        $result = $connection->executeQuery($query, ['Emails', $id, $this->id])->fetchOne();
        return $result !== false;
    }

    /**
     * Moves beans from one folder to another folder
     * @param string fromFolder GUID of source folder
     * @param string toFolder GUID of destination folder
     * @param string beanId GUID of SugarBean being moved
     */
    public function move($fromFolder, $toFolder, $beanId)
    {
        $this->db->getConnection()->update('folders_rel', ['folder_id' => $toFolder], [
            'folder_id' => $fromFolder,
            'polymorphic_id' => $beanId,
            'deleted' => 0,
        ]);
    }

    /**
     * Copies one bean from one folder to another
     */
    public function copyBean($fromFolder, $toFolder, $beanId, $module)
    {
        $this->db->getConnection()->insert('folders_rel', [
            'id' => Uuid::uuid1(),
            'folder_id' => $toFolder,
            'polymorphic_module' => $module,
            'polymorphic_id' => $beanId,
            'deleted' => 0,
        ]);
    }

    /**
     * Creates a new group Folder from the passed fields
     * @param array fields
     */
    public function setFolder($fields)
    {

        global $current_user;
        if (empty($fields['groupFoldersUser'])) {
            $fields['groupFoldersUser'] = $current_user->id;
        }

        $this->name = $fields['name'];
        $this->parent_folder = $fields['parent_folder'];
        $this->has_child = 0;
        $this->is_group = 1;
        $this->assign_to_id = $fields['groupFoldersUser'];
        $this->team_id = $fields['team_id'];
        $this->team_set_id = $fields['team_set_id'];

        $this->save();
    }

    /**
     * Returns GUIDs of folders that the user in focus is subscribed to
     * @param object user User object in focus
     * @return array
     */
    public function getSubscriptions($user)
    {
        if (empty($user)) {
            global $current_user;
            $user = $current_user;
        }
        return $this->db->getConnection()->executeQuery(
            'SELECT folder_id FROM folders_subscriptions WHERE assigned_user_id = ?',
            [$user->id]
        )->fetchFirstColumn();
    }

    /**
     * Sets a user's preferences for subscribe folders (Sugar only)
     * @param array subs Array of IDs for subscribed folders
     */
    public function setSubscriptions($subs)
    {
        global $current_user;

        if (empty($current_user->id)) {
            $GLOBALS['log']->fatal('*** FOLDERS: tried to update folder subscriptions for a user with no ID');
            return false;
        }

        $cleanSubscriptions = [];

        // ensure parent folders are selected, regardless.
        foreach ($subs as $id) {
            $id = trim($id);
            if (!empty($id)) {
                $cleanSubscriptions[] = $id;
                $parentFolder = $this->db->getConnection()
                    ->executeQuery('SELECT parent_folder FROM folders WHERE id = ?', [$id])
                    ->fetchOne();
                if (!empty($parentFolder)) {
                    $cleanSubscriptions = $this->getParentIDRecursive($parentFolder, $cleanSubscriptions);
                }
            }
        }

        $this->clearSubscriptions();

        foreach ($cleanSubscriptions as $id) {
            $this->insertFolderSubscription($id, $current_user->id);
        }
    }

    /**
     * Given a folder id and user id, create a folder subscription entry.
     *
     * @param String $folderId
     * @param String $userID
     * @return String The id of the newly created folder subscription.
     */
    public function insertFolderSubscription($folderId, $userID)
    {
        $id = Uuid::uuid1();
        $this->db->getConnection()->insert('folders_subscriptions', [
            'id' => $id,
            'folder_id' => $folderId,
            'assigned_user_id' => $userID,
        ]);
        return $id;
    }

    /**
     * Recursively finds parent node until it hits root
     * @param string id Starting id to follow up
     * @param array ret collected ids
     * @return array of IDs
     */
    public function getParentIDRecursive($id, $ret = [])
    {
        if (!safeInArray($id, $ret)) {
            $ret[] = $id;
        }

        $parentFolder = $this->db->getConnection()
            ->executeQuery('SELECT parent_folder FROM folders WHERE id = ? AND deleted = 0', [$id])->fetchOne();
        if (!empty($parentFolder)) {
            $ret = $this->getParentIDRecursive($parentFolder, $ret);
        }

        return $ret;
    }

    /**
     * Deletes subscriptions to folders in preparation for reset
     */
    public function clearSubscriptions()
    {
        global $current_user;

        if (!empty($current_user->id)) {
            $this->db->getConnection()
                ->delete('folders_subscriptions', ['assigned_user_id' => $current_user->id]);
        }
    }


    /**
     * Deletes all subscriptions for a particular folder id
     *
     * @return unknown
     */
    public function clearSubscriptionsForFolder($folderID)
    {
        $this->db->getConnection()->delete('folders_subscriptions', ['folder_id' => $folderID]);
    }

    protected function generateArchiveFolderQuery()
    {
        global $current_user;
        $q = <<<ENDQ
SELECT emails.id , emails.name, emails.date_sent, emails.status, emails.type, emails.flagged, emails.reply_to_status, emails_text.from_addr, emails_text.to_addrs, 'Emails' polymorphic_module FROM emails
JOIN emails_text on emails.id = emails_text.email_id
ENDQ;
        $this->emailBean->addVisibilityFrom($q, ['where_condition' => true]);

        $q .= <<<ENDW
 WHERE emails.deleted=0 AND emails.type NOT IN ('out', 'draft') AND emails.status NOT IN ('sent', 'draft') 
AND EXISTS (
SELECT 1 FROM emails_email_addr_rel eear
 JOIN email_addr_bean_rel eabr ON eabr.email_address_id=eear.email_address_id AND eabr.bean_id = '{$current_user->id}'
 AND eabr.bean_module = 'Users'
 WHERE eear.deleted=0 AND emails.id = eear.email_id
)
ENDW;
        $this->emailBean->addVisibilityWhere($q, ['where_condition' => true]);
        return $q;
    }

    public function generateSugarsDynamicFolderQuery()
    {
        global $current_user;
        $type = $this->folder_type;
        if ($type == 'archived') {
            return $this->generateArchiveFolderQuery();
        }
        $status = $type;
        if ($type == 'sent') {
            $type = 'out';
        }
        if ($type == 'inbound') {
            $ret = " AND emails.status NOT IN ('sent', 'archived', 'draft') AND emails.type NOT IN ('out', 'archived', 'draft')";
        } else {
            $ret = " AND emails.status NOT IN ('archived') AND emails.type NOT IN ('archived')";
        }
        $q = "SELECT
                emails.id,
                emails.name,
                emails.date_sent,
                emails.status,
                emails.type,
                emails.flagged,
                emails.reply_to_status,
                emails_text.from_addr,
                emails_text.to_addrs,
                'Emails' polymorphic_module
            FROM
                emails
            JOIN
                emails_text
            on
                emails.id = emails_text.email_id
                ";
        $options = [
            'where_condition' => true,
            'action' => 'list',
        ];
        $this->emailBean->addVisibilityFrom($q, $options);
        $q .= "
            WHERE
                (type = '{$type}' OR status = '{$status}')
                AND assigned_user_id = '{$current_user->id}'
                AND emails.deleted = 0 ";
        $this->emailBean->addVisibilityWhere($q, $options);
        return $q . $ret;
    } // fn

    /**
     * returns array of items for listView display in yui-ext Grid
     */
    public function getListItemsForEmailXML($folderId, $page = 1, $pageSize = 10, $sort = '', $direction = '')
    {
        global $timedate;
        global $current_user;
        global $beanList;
        global $sugar_config;
        global $app_strings;

        $this->retrieve($folderId);
        $start = ($page - 1) * $pageSize;

        $sort = (empty($sort)) ? $this->defaultSort : $sort;
        if (!in_array(strtolower($direction), ['asc', 'desc'])) {
            $direction = $this->defaultDirection;
        }

        if (!empty($this->hrSortLocal[$sort])) {
            $order = " ORDER BY {$this->hrSortLocal[$sort]} {$direction}";
        } else {
            $order = '';
        }

        if ($this->is_dynamic) {
            $r = $this->db->limitQuery(from_html($this->generateSugarsDynamicFolderQuery() . $order), $start, $pageSize);
        } else {
            // get items and iterate through them
            $q = "SELECT
                    emails.id,
                    emails.name,
                    emails.date_sent,
                    emails.status,
                    emails.type,
                    emails.flagged,
                    emails.reply_to_status,
                    emails_text.from_addr,
                    emails_text.to_addrs,
                    'Emails' polymorphic_module
                FROM
                    emails
                JOIN
                    folders_rel
                ON
                    emails.id = folders_rel.polymorphic_id
                JOIN
                    emails_text
                on
                    emails.id = emails_text.email_id
                    ";
            $this->emailBean->addVisibilityFrom($q, ['where_condition' => true]);
            $q .= "
                WHERE
                    folders_rel.folder_id = '{$folderId}'
                    AND folders_rel.deleted = 0
                    AND emails.deleted = 0 ";
            $this->emailBean->addVisibilityWhere($q, ['where_condition' => true]);
            if ($this->is_group) {
                $q = $q . " AND ((emails.assigned_user_id is null or emails.assigned_user_id = '') OR (emails.intent = 'createcase'))";
            }
            $r = $this->db->limitQuery($q . $order, $start, $pageSize);
        }

        $return = [];

        $email = BeanFactory::newBean('Emails'); //Needed for email specific functions.

        while ($a = $this->db->fetchByAssoc($r)) {
            $temp = [];
            $temp['flagged'] = (is_null($a['flagged']) || $a['flagged'] == '0') ? '' : 1;
            $temp['status'] = (is_null($a['reply_to_status']) || $a['reply_to_status'] == '0') ? '' : 1;
            $temp['from'] = preg_replace('/[\x00-\x08\x0B-\x1F]/', '', $a['from_addr']);
            $temp['subject'] = $a['name'];
            $temp['date'] = $timedate->to_display_date_time($this->db->fromConvert($a['date_sent'], 'datetime'));
            $temp['uid'] = $a['id'];
            $temp['mbox'] = 'sugar::' . $a['polymorphic_module'];
            $temp['ieId'] = $folderId;
            $temp['site_url'] = $sugar_config['site_url'];
            $temp['seen'] = ($a['status'] == 'unread') ? 0 : 1;
            $temp['type'] = $a['type'];
            $temp['hasAttach'] = $email->doesImportedEmailHaveAttachment($a['id']);
            $temp['to_addrs'] = preg_replace('/[\x00-\x08\x0B-\x1F]/', '', $a['to_addrs']);
            $return[] = $temp;
        }


        $metadata = [];
        $metadata['mbox'] = $app_strings['LBL_EMAIL_SUGAR_FOLDER'] . ': ' . $this->name;
        $metadata['ieId'] = $folderId;
        $metadata['name'] = $this->name;
        $metadata['unreadChecked'] = ($current_user->getPreference('showUnreadOnly', 'Emails') == 1) ? 'CHECKED' : '';
        $metadata['out'] = $return;

        return $metadata;
    }

    public function getCountItems($folderId)
    {
        global $current_user;
        global $beanList;
        global $sugar_config;
        global $app_strings;

        $this->retrieve($folderId);
        if ($this->is_dynamic) {
            $pattern = '/SELECT(.*?)(\s){1}FROM(\s){1}/is';  // ignores the case
            $replacement = 'SELECT count(*) c FROM ';
            $modified_select_query = preg_replace($pattern, $replacement, $this->generateSugarsDynamicFolderQuery(), 1);
            $r = $this->db->query(from_html($modified_select_query));
        } else {
            // get items and iterate through them
            $q = 'SELECT
                    count(*) c
                FROM
                    folders_rel
                JOIN
                    emails
                ON
                    emails.id = folders_rel.polymorphic_id
                    ';
            $this->emailBean->addVisibilityFrom($q, ['where_condition' => true]);
            $q .= "
                WHERE
                    folder_id = '{$folderId}'
                    AND folders_rel.deleted = 0
                    AND emails.deleted = 0 ";
            $this->emailBean->addVisibilityWhere($q, ['where_condition' => true]);
            if ($this->is_group) {
                $q .= " AND ((emails.assigned_user_id is null or emails.assigned_user_id = '') OR (emails.intent = 'createcase'))";
            }
            $r = $this->db->query($q);
        }

        $a = $this->db->fetchByAssoc($r);
        return $a['c'];
    }

    public function getCountUnread($folderId)
    {
        global $current_user;
        global $beanList;
        global $sugar_config;
        global $app_strings;

        $this->retrieve($folderId);
        if ($this->is_dynamic) {
            $pattern = '/SELECT(.*?)(\s){1}FROM(\s){1}/is';  // ignores the case
            $replacement = 'SELECT count(*) c FROM ';
            $modified_select_query = preg_replace($pattern, $replacement, $this->generateSugarsDynamicFolderQuery(), 1);
            $r = $this->db->query(from_html($modified_select_query) . " AND emails.status = 'unread'");
        } else {
            // get items and iterate through them
            $q = "SELECT
                    count(*) c
                FROM
                    folders_rel fr
                JOIN
                    emails
                on
                    fr.folder_id = '{$folderId}'
				";
            $this->emailBean->addVisibilityFrom($q, ['where_condition' => true]);
            $q .= "
                    AND fr.deleted = 0
                    AND fr.polymorphic_id = emails.id
                    AND emails.status = 'unread'
                    AND emails.deleted = 0 ";
            $this->emailBean->addVisibilityWhere($q, ['where_condition' => true]);
            if ($this->is_group) {
                $q .= " AND ((emails.assigned_user_id is null or emails.assigned_user_id = '') OR (emails.intent = 'createcase'))";
            }
            $r = $this->db->query($q);
        }

        $a = $this->db->fetchByAssoc($r);
        return $a['c'];
    }


    /**
     * Convenience method, pass a SugarBean and User to this to add anything to a given folder
     */
    public function addBean($bean, $user = null)
    {
        if (empty($bean->id) || empty($bean->module_dir)) {
            $GLOBALS['log']->fatal('*** FOLDERS: addBean() got empty bean - not saving');
            return false;
        } elseif (empty($this->id)) {
            $GLOBALS['log']->fatal('*** FOLDERS: addBean() is trying to save to a non-saved or non-existent folder');
            return false;
        }

        $this->db->getConnection()->insert('folders_rel', [
            'id' => Uuid::uuid1(),
            'folder_id' => $this->id,
            'polymorphic_module' => $bean->module_dir,
            'polymorphic_id' => $bean->id,
            'deleted' => 0,
        ]);

        return true;
    }

    /**
     * Builds up a metacollection of user/group folders to be passed to processor methods
     * @param object User object, defaults to $current_user
     * @return array Array of abstract folder objects
     */
    public function retrieveFoldersForProcessing($user, $subscribed = true)
    {
        global $sugar_config;
        global $current_language, $current_user;

        $emails_mod_strings = return_module_language($current_language, 'Emails');
        $myEmailTypeString = 'inbound';
        $myDraftsTypeString = 'draft';
        $mySentEmailTypeString = 'sent';

        if (empty($user)) {
            global $current_user;
            $user = $current_user;
        }
        $teamSecurityClause = '';

        $bean = new SugarBean();
        $bean->disable_row_level_security = false;
        $bean->add_team_security_where_clause($teamSecurityClause, 'f');
        $bean->disable_row_level_security = true;

        // need space in coalesce for oracle to avoid to null conversion
        $rootWhere = "AND (f.parent_folder IS NULL OR f.parent_folder = '')";
        $parameters = [];
        if ($subscribed) {
            $q = $this->coreSubscribed . $teamSecurityClause . $this->coreWhereSubscribed . '? '
                . $rootWhere . $this->coreOrderBy;
            $parameters[] = $user->id;
        } else {
            $q = $this->core . $teamSecurityClause . $this->coreWhere . $rootWhere . $this->coreOrderBy;
        }
        $stmt = $this->db->getConnection()->executeQuery($q, $parameters);
        $return = [];

        $found = [];
        while ($a = $stmt->fetchAssociative()) {
            $a['created_by'] = $this->db->fromConvert($a['created_by'], 'id');

            if ((($a['folder_type'] == $myEmailTypeString) ||
                    ($a['folder_type'] == $myDraftsTypeString) ||
                    ($a['folder_type'] == $mySentEmailTypeString)) &&
                ($a['created_by'] != $current_user->id)) {
                continue;
            } // if
            if (!isset($found[$a['id']])) {
                $found[$a['id']] = true;
                $return[] = $a;
            }
        }
        return $return;
    }

    /**
     * Preps object array for async call from user's Settings->Folders
     */
    public function getGroupFoldersForSettings($focusUser = null)
    {
        global $app_strings;

        $grp = [];

        $folders = $this->retrieveFoldersForProcessing($focusUser, false);
        $subscriptions = $this->getSubscriptions($focusUser);

        foreach ($folders as $a) {
            $a['selected'] = (safeInArray($a['id'], $subscriptions)) ? true : false;
            $a['origName'] = $a['name'];

            if ($a['is_group'] == 1) {
                if ($a['deleted'] != 1) {
                    $grp[] = $a;
                }
            }
        }

        return $grp;
    }

    /**
     * Preps object array for async call from user's Settings->Folders
     */
    public function getFoldersForSettings($focusUser = null)
    {
        global $app_strings;

        $user = [];
        $grp = [];
        $user[] = ['id' => '', 'name' => $app_strings['LBL_NONE'], 'has_child' => 0, 'is_group' => 0, 'selected' => false];
        $grp[] = ['id' => '', 'name' => $app_strings['LBL_NONE'], 'has_child' => 0, 'is_group' => 1, 'selected' => false, 'origName' => ''];

        $folders = $this->retrieveFoldersForProcessing($focusUser, false);
        $subscriptions = $this->getSubscriptions($focusUser);

        foreach ($folders as $a) {
            $a['selected'] = (safeInArray($a['id'], $subscriptions)) ? true : false;
            $a['origName'] = $a['name'];
            if (isset($a['dynamic_query'])) {
                unset($a['dynamic_query']);
            }
            if ($a['is_group'] == 1) {
                $grp[] = $a;
            } else {
                $user[] = $a;
            }

            if ($a['has_child'] == 1) {
                $getChildrenStmt = $this->db->getConnection()->executeQuery(
                    $this->core . $this->coreWhere . 'AND parent_folder = ?',
                    [$a['id']]
                );
                while ($aGetChildren = $getChildrenStmt->fetchAssociative()) {
                    if ($a['is_group']) {
                        $this->_depth = 1;
                        $grp = $this->getFoldersChildForSettings($aGetChildren, $grp, $subscriptions);
                    } else {
                        $this->_depth = 1;
                        $user = $this->getFoldersChildForSettings($aGetChildren, $user, $subscriptions);
                    }
                }
            }
        }

        $ret = [
            'userFolders' => $user,
            'groupFolders' => $grp,
        ];
        return $ret;
    }

    public function getFoldersChildForSettings($a, $collection, $subscriptions)
    {
        $a['selected'] = (safeInArray($a['id'], $subscriptions)) ? true : false;
        $a['origName'] = $a['name'];

        if (isset($a['dynamic_query'])) {
            unset($a['dynamic_query']);
        }

        for ($i = 0; $i < $this->_depth; $i++) {
            $a['name'] = '.' . $a['name'];
        }

        $collection[] = $a;

        if ($a['has_child'] == 1) {
            $this->_depth++;
            $getChildrenStmt = $this->db->getConnection()->executeQuery(
                $this->core . $this->coreWhere . 'AND parent_folder = ?',
                [$a['id']]
            );
            while ($aGetChildren = $getChildrenStmt->fetchAssociative()) {
                $collection = $this->getFoldersChildForSettings($aGetChildren, $collection, $subscriptions);
            }
        }

        return $collection;
    }

    /**
     * Returns the number of "new" items (based on passed criteria)
     * @param string id ID of folder
     * @param array criteria
     *      expected:
     *      array('field' => 'status',
     *              'value' => 'unread');
     * @param array
     * @return int
     */
    public function getCountNewItems($id, $criteria, $folder)
    {
        global $current_user;

        $sugarFolder = new SugarFolder();
        return $sugarFolder->getCountUnread($id);
    }

    /**
     * Collects, sorts, and builds tree of user's folders
     * @param objec $rootNode Reference to tree root node
     * @param array $folderStates User pref folder open/closed states
     * @param object $user Optional User in focus, default current_user
     * @return array
     */
    public function getUserFolders(&$rootNode, $folderStates, $user = null, $forRefresh = false)
    {
        if (empty($user)) {
            global $current_user;
            $user = $current_user;
        }
        global $mod_strings;
        $folders = $this->retrieveFoldersForProcessing($user, true);
        $subscriptions = $this->getSubscriptions($user);

        $refresh = ($forRefresh) ? [] : null;

        if (!is_array($folderStates)) {
            $folderStates = [];
        }

        foreach ($folders as $a) {
            if ($a['deleted'] == 1) {
                continue;
            }
            $label = ($a['name'] == 'My Email' ? $mod_strings['LNK_MY_INBOX'] : $a['name']);

            $unseen = $this->getCountNewItems($a['id'], ['field' => 'status', 'value' => 'unread'], $a);

            $folderNode = new ExtNode($a['id'], $label);
            $folderNode->dynamicloadfunction = '';
            $folderNode->expanded = false;

            if (array_key_exists('Home::' . $a['id'], $folderStates)) {
                if ($folderStates['Home::' . $a['id']] == 'open') {
                    $folderNode->expanded = true;
                }
            }
            $nodePath = 'Home::' . $folderNode->_properties['id'];

            $folderNode->dynamic_load = true;
            $folderNode->set_property('ieId', 'folder');
            $folderNode->set_property('is_group', ($a['is_group'] == 1) ? 'true' : 'false');
            $folderNode->set_property('is_dynamic', ($a['is_dynamic'] == 1) ? 'true' : 'false');
            $folderNode->set_property('mbox', $folderNode->_properties['id']);
            $folderNode->set_property('unseen', $unseen);
            $folderNode->set_property('id', $a['id']);
            $folderNode->set_property('folder_type', $a['folder_type']);
            $folderNode->set_property('children', []);

            if (safeInArray($a['id'], $subscriptions) && $a['has_child'] == 1) {
                $getChildrenStmt = $this->db->getConnection()->executeQuery(
                    $this->core . $this->coreWhere . 'AND parent_folder = ?',
                    [$a['id']]
                );
                while ($aGetChildren = $getChildrenStmt->fetchAssociative()) {
                    if (safeInArray($aGetChildren['id'], $subscriptions)) {
                        $folderNode->add_node($this->buildTreeNodeFolders($aGetChildren, $nodePath, $folderStates, $subscriptions));
                    }
                }
            }
            $rootNode->add_node($folderNode);
        }

        /* the code below is called only by Settings->Folders when selecting folders to subscribe to */
        if ($forRefresh) {
            $metaNode = [];

            if (!empty($rootNode->nodes)) {
                foreach ($rootNode->nodes as $node) {
                    $metaNode[] = $this->buildTreeNodeRefresh($node, $subscriptions);
                }
            }
            return $metaNode;
        }
    }

    /**
     * Builds up a metanode for folder refresh (Sugar folders only)
     */
    public function buildTreeNodeRefresh($folderNode, $subscriptions)
    {
        $metaNode = $folderNode->_properties;
        $metaNode['expanded'] = $folderNode->expanded;
        $metaNode['text'] = $folderNode->_label;
        if ($metaNode['is_group'] == 'true') {
            $metaNode['cls'] = 'groupFolder';
        } else {
            $metaNode['cls'] = 'sugarFolder';
        }
        $metaNode['id'] = $folderNode->_properties['id'];
        $metaNode['children'] = [];
        $metaNode['type'] = 1;
        $metaNode['leaf'] = false;
        $metaNode['isTarget'] = true;
        $metaNode['allowChildren'] = true;

        if (!empty($folderNode->nodes)) {
            foreach ($folderNode->nodes as $node) {
                if (safeInArray($node->_properties['id'], $subscriptions)) {
                    $metaNode['children'][] = $this->buildTreeNodeRefresh($node, $subscriptions);
                }
            }
        }
        return $metaNode;
    }

    /**
     * Builds children nodes for folders for TreeView
     * @return $folderNode TreeView node
     */
    public function buildTreeNodeFolders($a, $nodePath, $folderStates, $subscriptions)
    {
        $label = $a['name'];
        global $mod_strings;
        if ($a['name'] == 'My Drafts') {
            $label = $mod_strings['LBL_LIST_TITLE_MY_DRAFTS'];
        }
        if ($a['name'] == 'Sent Emails') {
            $label = $mod_strings['LBL_LIST_TITLE_MY_SENT'];
        }
        $unseen = $this->getCountNewItems($a['id'], ['field' => 'status', 'value' => 'unread'], $a);

        $folderNode = new ExtNode($a['id'], $label);
        $folderNode->dynamicloadfunction = '';
        $folderNode->expanded = false;

        $nodePath .= "::{$a['id']}";

        if (array_key_exists($nodePath, $folderStates)) {
            if ($folderStates[$nodePath] == 'open') {
                $folderNode->expanded = true;
            }
        }

        $folderNode->dynamic_load = true;
        $folderNode->set_property('click', "SUGAR.email2.listView.populateListFrameSugarFolder(YAHOO.namespace('frameFolders').selectednode, '{$a['id']}', 'false');");
        $folderNode->set_property('ieId', 'folder');
        $folderNode->set_property('mbox', $a['id']);
        $folderNode->set_property('is_group', ($a['is_group'] == 1) ? 'true' : 'false');
        $folderNode->set_property('is_dynamic', ($a['is_dynamic'] == 1) ? 'true' : 'false');
        $folderNode->set_property('unseen', $unseen);
        $folderNode->set_property('folder_type', $a['folder_type']);

        if (safeInArray($a['id'], $subscriptions) && $a['has_child'] == 1) {
            $getChildrenStmt = $this->db->getConnection()->executeQuery(
                $this->core . $this->coreWhere . 'AND parent_folder = ? ' . $this->coreOrderBy,
                [$a['id']]
            );

            while ($aGetChildren = $getChildrenStmt->fetchAssociative()) {
                $folderNode->add_node($this->buildTreeNodeFolders($aGetChildren, $nodePath, $folderStates, $subscriptions));
            }
        }
        return $folderNode;
    }

    /**
     * Flags a folder as deleted
     * @return bool True on success
     */
    public function delete()
    {
        global $current_user;

        if (!empty($this->id)) {
            if ($this->has_child) {
                $this->deleteChildrenCascade($this->id);
            }
            $qb = $this->db->getConnection()->createQueryBuilder();
            $qb->update('folders')
                ->set('deleted', 1)
                ->where($qb->expr()->eq('id', $qb->createPositionalParameter($this->id)));
            if (!$current_user->isAdmin()) {
                $qb->andWhere($qb->expr()->eq('created_by', $current_user->id));
            }
            $qb->execute();
            return true;
        }
        return false;
    }

    /**
     * Deletes all children in a cascade
     * @param string $id ID of parent
     * @return bool True on success
     */
    public function deleteChildrenCascade($id)
    {
        global $current_user;
        $conn = $this->db->getConnection();
        $platform = $conn->getDatabasePlatform();
        $query = $platform->modifyLimitQuery(
            'SELECT NULL FROM inbound_email WHERE groupfolder_id = ? and deleted = 0',
            1
        );
        $result = $conn->executeQuery($query, [$id])->fetchOne();
        if ($result !== false) {
            return false;
        } // if
        $query = $platform->modifyLimitQuery(
            'SELECT NULL FROM folders_rel where polymorphic_module = ? and folder_id = ? and deleted = 0',
            1
        );
        $result = $conn->executeQuery($query, ['Emails', $id])->fetchOne();
        if ($result !== false) {
            return false;
        } // if
        $doesFolderHaveChild = $conn->executeQuery('SELECT has_child FROM folders WHERE id = ?', [$id])->fetchOne();
        $canContinue = true;
        if ($doesFolderHaveChild == 1) {
            $subFolderStmt = $conn->executeQuery(
                'SELECT id FROM folders WHERE parent_folder = ?',
                [$id]
            );
            while ($subFolder = $subFolderStmt->fetchAssociative()) {
                $canContinue = $this->deleteChildrenCascade($subFolder['id']);
            }
        }
        if ($canContinue) {
            // flag deleted
            $update = $conn->createQueryBuilder()
                ->update('folders')
                ->set('deleted', 1);
            $update->where($update->expr()->eq('id', $update->createPositionalParameter($id)));
            if (!$current_user->isAdmin()) {
                $update->andWhere(
                    $update->expr()->eq('created_by', $update->createPositionalParameter($current_user->id))
                );
            }
            $update->execute();
            // flag rels
            $conn->update('folders_rel', ['deleted' => 1], ['folder_id' => $id]);
            // delete subscriptions
            $conn->delete('folders_subscriptions', ['folder_id' => $id]);
        }

        return $canContinue;
    }

    /**
     * Return the default mapping values.
     *
     * @return array Mapping key-value pairs.
     */
    protected function getFieldMapping()
    {
        global $current_user;
        return [
            'created_by' => $current_user->id,
            'modified_by' => $current_user->id,
            'dynamic_query' => $this->dynamic_query,
            'deleted' => 0,
        ];
    }

    /**
     * return DB field values according to mapping and object properties
     * @param array $fields
     * @return array
     */
    protected function getFieldValues(array $fields)
    {
        $result = [];
        $mapping = $this->getFieldMapping();
        foreach ($fields as $name) {
            if (array_key_exists($name, $this->fields)) {
                switch (true) {
                    case isset($this->$name):
                        $result[$name] = $this->$name;
                        break;
                    case array_key_exists($name, $mapping):
                        $result[$name] = $mapping[$name];
                        break;
                    default:
                        $result[$name] = null;
                }
            }
        }
        return $result;
    }

    /**
     * Saves folder
     * @return bool
     */
    public function save($addSubscriptions = true)
    {

        if ((empty($this->id) && $this->new_with_id == false) || (!empty($this->id) && $this->new_with_id == true)) {
            if (empty($this->id)) {
                $this->id = Uuid::uuid1();
            }
            $this->db->insertParams($this->table, $this->fields, $this->getFieldValues(array_keys($this->fields)));

            if ($addSubscriptions) {
                // create default subscription
                $this->addSubscriptionsToGroupFolder();
            }
            $this->db->getConnection()->update($this->table, ['has_child' => 1], ['id' => $this->parent_folder]);
        } else {
            $values = $this->getFieldValues(array_keys($this->fields));
            unset($values['id'], $values['created_by']);
            $this->db->updateParams($this->table, $this->fields, $values, ['id' => $this->id]);
        }
        return true;
    }

    /**
     * Add subscriptions to this group folder.
     *
     */
    public function addSubscriptionsToGroupFolder()
    {
        global $current_user;

        $this->createSubscriptionForUser($current_user->id);

        if ($this->is_group) {
            $team = BeanFactory::getBean('Teams', $this->team_id);
            $usersList = $team->get_team_members(true);
            foreach ($usersList as $userObject) {
                $this->createSubscriptionForUser($userObject->id);
            }
        }
    }


    /**
     * Add subscriptions to this group folder.
     *
     */
    public function createSubscriptionForUser($user_id)
    {
        $this->db->insertParams(
            $this->folderSubscriptionTable,
            $this->folderSubscriptionFields,
            [
                'id' => Uuid::uuid1(),
                'folder_id' => $this->id,
                'assigned_user_id' => $user_id,
            ]
        );
    }

    /**
     * @param array $fields
     * @return array
     * @throws DBALException
     */
    public function updateFolder($fields)
    {
        $this->dynamic_query = $this->db->quote($this->dynamic_query);
        $id = $fields['record'];
        $name = $fields['name'];
        $parent_folder = $fields['parent_folder'];
        $team_id = $fields['team_id'];
        $team_set_id = $fields['team_set_id'];
        // first do the retrieve
        $this->retrieve($id);
        if ($this->has_child) {
            $childrenArray = [];
            $this->findAllChildren($id, $childrenArray);
            if (safeInArray($parent_folder, $childrenArray)) {
                return ['status' => 'failed', 'message' => 'Can not add this folder to its children'];
            }
        }
        // update has_child to 0 for this parent folder if this is the only child it has
        $connection = $this->db->getConnection();
        $platform = $connection->getDatabasePlatform();
        $query = $platform->modifyLimitQuery(
            sprintf('SELECT NULL FROM %s WHERE parent_folder = ? AND deleted = 0', $this->table),
            2
        );
        $result = $connection->executeQuery($query, [$this->parent_folder])->fetchFirstColumn();
        if (safeCount($result) == 1) {
            $this->db->getConnection()->update($this->table, ['has_child' => 0], ['id' => $this->parent_folder]);
        } // if
        $this->name = $name;
        $this->parent_folder = $parent_folder;
        $this->team_id = $team_id;
        $this->team_set_id = $team_set_id;
        $values = $this->getFieldValues([
            'dynamic_query',
            'parent_folder',
            'team_set_id',
            'team_id',
            'modified_by',
        ]);
        $this->db->updateParams($this->table, $this->fields, $values, ['id' => $this->id]);
        if (!empty($this->parent_folder)) {
            $this->db->getConnection()->update($this->table, ['has_child' => 1], ['id' => $this->parent_folder]);
        } // if
        return ['status' => 'done'];
    } // fn

    public function findAllChildren($folderId, &$childrenArray)
    {
        $conn = $this->db->getConnection();
        $doesFolderHaveChild = $conn
            ->executeQuery('SELECT has_child FROM folders WHERE id = ?', [$folderId])
            ->fetchOne();
        if ($doesFolderHaveChild == 1) {
            $subFolderStmt = $conn->executeQuery(
                'SELECT id FROM folders WHERE parent_folder = ? AND deleted = 0',
                [$folderId]
            );
            while ($a2 = $subFolderStmt->fetchAssociative()) {
                $childrenArray[] = $a2['id'];
                $this->findAllChildren($a2['id'], $childrenArray);
            } // while
        } // if
    } // fn

    /**
     * Retrieves and populates object
     * @param string $id ID of folder
     * @return bool True on success
     */
    public function retrieve($id)
    {
        $data = $this->db->getConnection()->executeQuery(
            'SELECT * FROM folders WHERE id = ? AND deleted = 0',
            [$id]
        )->fetchAssociative();
        if ($data) {
            foreach ($data as $name => $value) {
                $this->$name = $this->db->fromConvert($value, $this->fields[$name]['type']);
            }
            return true;
        } else {
            return false;
        }
    }
}
